#!/usr/bin/perl -w

#/***************************************************************************
# *   Calculation of Pi to arbitrary decimal places			    *
# *   calcpi.pl - a Perl script for automatically distributing the task	    *
# *   of calculating pi across multiple UNIX systems			    *
# *									    *
# *   Copyright (C) 2005 by Musa A. Maharramov   			    *
# *									    *
# *   Department of Applied Mathematics and Cybernetics			    * 
# *   Baku State University						    *
# *   musa@maharramov.com						    *
# *   www.maharramov.com						    *
# *						  			    *
# *   Permission is hereby granted, free of charge, to any person obtaining *
# *   a copy of this software and associated documentation files (the       *
# *   "Software"), to deal in the Software without restriction, including   *
# *   without limitation the rights to use, copy, modify, merge, publish,   *
# *   distribute, sublicense, and/or sell copies of the Software, and to    *
# *   permit persons to whom the Software is furnished to do so, subject to *
# *   the following conditions:                                             *
# *                                                                         *
# *   The above copyright notice and this permission notice shall be        *
# *   included in all copies or substantial portions of the Software.       *
# *                                                                         *
# *   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,       *
# *   EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF    *
# *   MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.*
# *   IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR     *
# *   OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, *
# *   ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR *
# *   OTHER DEALINGS IN THE SOFTWARE.                                       *
#***************************************************************************/

#/*
#	calcpi.pl - a Perl script for automatically distributing the task 
#		of calculating pi across multiple UNIX systems
#	
#	Other project files include:
#
#	pi.cpp - the master file of the program calculting Pi
#		or a specified part of the corresponding series
#	pi.h - master header file
#	strings.cpp - separately compiled string message file
#		messagestrings.h - corresponding declaration file 
#	longmath.h - high precision arithmetic header file
#	longmath.cpp - separately compiled high precision arithmetic library
#	longadd.cpp - a standalone program for adding two high-precision numbers
#	ehosts.txt - a text file with a list of hosts to be used for performing 
#		 the calculation (similar to e.g. LAM schema file)
#	Makefile - make file for GNU make
#
#*/

use	POSIX qw(strtod); # string to number conversion - MM 

$ComputeNodes = 'ehosts.txt';	# default compute nodes file - MM
$User = "";		# default user account for running remote execute commands is the current one - MM
$Exec = 'ssh -x';	# default execute command is ssh -x - MM 
$CurrDir = `pwd`;
chomp $CurrDir;
$Progname = "$CurrDir/pi";	# default pi calculator - MM
$AddProgname = "$CurrDir/longadd"; # and default "long add" programme - MM
$HelpText = "\tUsage: calcpi.pl [-f <filename>] [-u <username>] [-exec <execute command>] \
[-p <programme name>] [-la <long-add programme name>] [-v {0,1}] [-l {0,1}] \
[-lf <log filename>] [-d <4-100000000>] [-h]\n";
$bVerbose = 0;		# be verbose - MM
$bLog = 0; 		# create execution log - MM
$LogFile = "calcpi.log";
$Digits = 100;
$NoOfTermsToSum = 100;
@NodeNames = ();	# names of the compute nodes - MM

# message logging - MM
sub logmsg($) {
  my $s = shift;
  if ($bLog) {print LOG "CALCPI: $s\n";}# output to log file - MM
  if ($bVerbose) {print "$s\n";} 	# output to STDOUT if verbose - MM
}

# command-line args
%param = (
	'-f' => sub {$ComputeNodes = shift;},	# compute node (hosts) file - MM
	'-u' => sub {$User = shift;},		# user account for running remote execute commands - MM
	'-exec' => sub {$Exec = shift;},	# remote execute command - MM
	'-p' => sub{$Progname = shift;},	# name of the programme for calculating pi - MM
	'-la' => sub{$AddProgname = shift;},	# name of the programme for adding long numbers - MM	
	'-h' => sub {print $HelpText; exit(1);},
	'-v' => sub {$bVerbose = shift;},
	'-l' => sub {$bLog = shift;},
	'-lf' => sub {$LogFile = shift;},
	'-d' => sub {	$Digits = shift; 
			defined($Digits) && $Digits >= 4 && $Digits <= 100000000 or 
				die "No of digits must be 4-100000000\n"; 
		}
);

@computers = (); # list of compute nodes read off the schema file - MM

# parse command-line arguments - MM

while ($_=shift) {
	if (defined($param{$_}) && defined($parvalue = shift))  {
		&{$param{$_}}($parvalue);
	}
	else {
 		print "unrecognised option $_;\n\n";
  		&{$param{'-h'}}();
	}
}

if ($bLog) {
	open (LOG, "+>$LogFile") or 
		die "failure opening the log file $LogFile\n";
};

logmsg "Starting calculation of Pi on ".`date`."";

logmsg "Success parsing command-line parameters:\n\tComputeNodes=$ComputeNodes\n\tUser=$User\n\tExec=$Exec\n\tProgname=$Progname\n\tAddProgname=$AddProgname\n\tVerbose=$bVerbose\n\tbLog=$bLog\n\tLogFile=$LogFile\n\tDigits=$Digits";

# output of ./pi -d <digits> -r command converted from string to number - MM
($NoOfTermsToSum = eval "`".$Progname." -d ".$Digits." -r `")  && defined($NoOfTermsToSum = strtod($NoOfTermsToSum)) or 
	close (LOG) && die "failure identifying the number of terms required to sum\n";

logmsg "summing $NoOfTermsToSum terms of the series";
logmsg "opening the compute nodes (schema) file $ComputeNodes";

open (NODES, "< $ComputeNodes") or 
	logmsg "failure opening compute nodes (schema) file $ComputeNodes" &&
	close LOG && die "failure opening compute nodes (schema) file $ComputeNodes";


while ($_=<NODES>) {
	chomp;
	my $node = $_;
	next unless ($node);
	logmsg "adding compute node $node";
	push @computers, "$Exec ".($User ? "$User@" : "").$node." \"$Progname -d $Digits "; 
	push @NodeNames, $node;
}
close NODES;

if (!@NodeNames) {
	logmsg "no compute nodes defined - quitting";
	close LOG;
	die;
}

# distribute the workload among the nodes - MM
$startindex=0;
unless ($delta = int($NoOfTermsToSum/@NodeNames)) {$delta = 1;}


# distribute the task evenly among Card(@NodeNames) compute nodes - MM
foreach ($i=0;$i<@NodeNames-1;$i++) {
	$computers[$i] .= "-s $startindex -n $delta > ~/$NodeNames[$i]_temp_pi_".$i."\"";
	logmsg "prepared command for remote exec: ".$computers[$i];
	$startindex += $delta;
}

if ($startindex<$NoOfTermsToSum) {
	$computers[@NodeNames-1] .= "-s $startindex -n ".($NoOfTermsToSum-$startindex)." > ~/$NodeNames[@NodeNames-1]_temp_pi_".(@NodeNames-1)."\"";
	logmsg "prepared command for remote exec: ".$computers[@NodeNames-1];
}

# now starting the actual distributed calculation - MM

@pid = ();

$iCurrentProcess = 0; 
while ($iCurrentProcess < @NodeNames) {
	if ($pid[$iCurrentProcess] = fork()) {
		logmsg "running command ".$computers[$iCurrentProcess];
		$iCurrentProcess++;
	}
	else {exec($computers[$iCurrentProcess]);}
}

# now wait for all prpocesses to finish - MM
foreach ($i=0;$i<@NodeNames;$i++) {
	waitpid($pid[$i], 0);
	logmsg "Node ".$NodeNames[$i]." finished";
}
logmsg "all partial calculation processes finished";

# now sum up the partial results - MM
$longaddcmd = "$AddProgname -d $Digits";
foreach ($i=0;$i<@NodeNames;$i++) {
	$longaddcmd = $longaddcmd." -f ~/$NodeNames[$i]_temp_pi_$i";
}

$longaddcmd = $longaddcmd." > pi_with_"."$Digits"."_dec_places";
logmsg "long add command prepared: $longaddcmd";

($addprocess = fork) ? waitpid($addprocess, 0) : exec($longaddcmd);
logmsg "longadd finished";
logmsg "Done on ".`date`;
	
close LOG;
