/***************************************************************************
 *   HIGH-PRECISION FIXED-POINT REAL ARITHMETIC CLASS LIBRARY		   *
 *   Version 0.5 Beta							   *
 *									   * 
 *   Copyright (C) 2005 by Musa A. Maharramov   			   *
 *                                                                         *
 *   Department of Applied Mathematics and Cybernetics			   * 
 *   Baku State University						   *
 *   musa@maharramov.com						   *
 *   www.maharramov.com							   *
 *                                                                         *
 *   This library has been specifically developed for high-precision 	   *
 *   calculation of PI and E.						   *
 * 									   *
 *   Implemented arithmetic operation include summation, subtraction 	   *
 *   and negation of high-precision fixed-point real numbers, and 	   *
 *   division by long signed integers.					   *
 *									   *
 *						  			   *
 *   Permission is hereby granted, free of charge, to any person obtaining *
 *   a copy of this software and associated documentation files (the       *
 *   "Software"), to deal in the Software without restriction, including   *
 *   without limitation the rights to use, copy, modify, merge, publish,   *
 *   distribute, sublicense, and/or sell copies of the Software, and to    *
 *   permit persons to whom the Software is furnished to do so, subject to *
 *   the following conditions:                                             *
 *                                                                         *
 *   The above copyright notice and this permission notice shall be        *
 *   included in all copies or substantial portions of the Software.       *
 *                                                                         *
 *   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,       *
 *   EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF    *
 *   MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.*
 *   IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR     *
 *   OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, *
 *   ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR *
 *   OTHER DEALINGS IN THE SOFTWARE.                                       *
 ***************************************************************************/


// the following include file is ignored for UNIX-like systems
// and is required for the generation of Microsoft pre-compiled headers - MM

#ifdef _WIN32			
	#include "stdafx.h"
#endif

#include "longmath.h"
#include <stdio.h>
#include <stdlib.h>

// uncomment for verbose memory management - MM
//#define DEBUG

cLNUMBER::cLNUMBER (unsigned int uNoOfDigits) throw(char*) : uNoOfAtoms(0) , pData(NULL), bGarbage(false) {
	pData = (unsigned long*)malloc((uNoOfAtoms = (1 + uNoOfDigits/NOOFDIGS + ((uNoOfDigits % NOOFDIGS) != 0 ? 1 : 0)))*sizeof(unsigned long));
	if (!pData) {uNoOfAtoms=0; throw("Failure allocating memory for cLNUMBER");}
	memset((void*)pData, 0, uNoOfAtoms*sizeof(unsigned long));
}

cLNUMBER::cLNUMBER (const cLNUMBER& Num)  throw(char*) : uNoOfAtoms(0) , pData(NULL), bGarbage(false) {
	pData = (unsigned long*) malloc((uNoOfAtoms=Num.uNoOfAtoms)*sizeof(unsigned long));
	if (!pData) {uNoOfAtoms=0; throw("Failure allocating memory for cLNUMBER");}
	memcpy((void*)pData, (void*)Num.pData, uNoOfAtoms*sizeof(unsigned long));
}

cLNUMBER& cLNUMBER::operator+(cLNUMBER& Num)  throw(char*) {
	cLNUMBER* pNewNum = new cLNUMBER(*this);
	// add extra digits to match accuracy of the addant - MM
	if (pNewNum->uNoOfAtoms < Num.uNoOfAtoms) { 
		unsigned long *p2 = (unsigned long*) malloc(Num.uNoOfAtoms*sizeof(unsigned long));
		if (!p2) throw("Failure allocating memory for cLNUMBER");
		memcpy((void*)p2, (void*)pData, Num.uNoOfAtoms*sizeof(unsigned long));
		free(pNewNum->pData);
		pNewNum->pData = p2;
		pNewNum->uNoOfAtoms = Num.uNoOfAtoms;
	}

	// now add the other number to this cLNUMBER - MM
	int uCarryOver=0;
	for (int i=Num.uNoOfAtoms-1;i>=0;i--) {
		pNewNum->pData[i] = (pData[i]+Num.pData[i]+uCarryOver) % BASE;
		uCarryOver = (pData[i]+Num.pData[i]+uCarryOver)/BASE;
	}
	// mark the temporary object for release by our custom garbage collector - MM
	pNewNum->bGarbage = true; 
	// self-destruct if self is a dynamically created unrefereced object - MM
	if (bGarbage) {
#ifdef DEBUG
		printf("destroying self after adding another...\n");
#endif
		ReleaseMemory();
	}
	if (Num.bGarbage) {
#ifdef DEBUG
		printf("destroying the other after adding it...\n");
#endif
		Num.ReleaseMemory();
	}

	return *pNewNum;
}

cLNUMBER& cLNUMBER::operator-(cLNUMBER& Num)  throw(char*) {
	cLNUMBER* pNewNum = new cLNUMBER(*this);
	// add extra digits to match accuracy of the addant - MM
	if (pNewNum->uNoOfAtoms < Num.uNoOfAtoms) { 
		unsigned long *p2 = (unsigned long*) malloc(Num.uNoOfAtoms*sizeof(unsigned long));
		if (!p2) throw("Failure allocating memory for cLNUMBER");
		memcpy((void*)p2, (void*)pData, Num.uNoOfAtoms*sizeof(unsigned long));
		free(pNewNum->pData);
		pNewNum->pData = p2;
		pNewNum->uNoOfAtoms = Num.uNoOfAtoms;
	}

	// now subtract Num from this cLNUMBER - MM
	cLNUMBER negated(Num);
	negated.Negate();
	*pNewNum = *pNewNum + negated;
	/* mark the temporary object for ReleaseMemory by our custom garbage collector;
	 do not set this flag before the previous line though - 
	 otherwise the operator + will ReleaseMemory pNewNum's data - MM */
	pNewNum->bGarbage = true;
	// self-destruct if self is a dynamically created unrefereced object - MM
	if (bGarbage) {
#ifdef DEBUG
		printf("destroying self after subtracting another...\n");
#endif
		ReleaseMemory();
	}
	if (Num.bGarbage) {
#ifdef DEBUG
		printf("destroying the other after subtracting it...\n");
#endif
		Num.ReleaseMemory();
	}

	return *pNewNum;
}



cLNUMBER& cLNUMBER::operator/(int Num)  throw(char*){
	if (Num == 0) throw("Division by zero");
	
	cLNUMBER* pNewNum = new cLNUMBER(*this);
	unsigned temp = 0;
	int uCarryOver=0, sign = pNewNum->iSign();
	
	// always divide a +ive by a +ive then correct sign - MM
	if (sign == -1) pNewNum->Negate();
	if (Num < 0) {Num = -Num; sign = -sign;}

	for (unsigned int i=0; i<uNoOfAtoms;i++) {
		temp = pNewNum->pData[i];
		pNewNum->pData[i] = (temp + uCarryOver*BASE) / Num;
		uCarryOver = (temp + uCarryOver*BASE) % Num;
	}
	if (sign == -1) pNewNum->Negate();
	// mark the temporary object for ReleaseMemory by our custom garbage collector - MM
	pNewNum->bGarbage = true;
	// self-destruct if self is a dynamically created unrefereced object - MM
	if (bGarbage) {
#ifdef DEBUG
		printf("destroying self after dividing by another...\n");
#endif
		ReleaseMemory();
	}
	return *pNewNum;
}

cLNUMBER& cLNUMBER::operator=(cLNUMBER& Num)  throw(char*){
	if (this == &Num) return *this;
	// add extra digits to match accuracy of the addant - MM
	if (uNoOfAtoms < Num.uNoOfAtoms) { 
		unsigned long *p2 = (unsigned long*) malloc(Num.uNoOfAtoms*sizeof(unsigned long));
		if (!p2) throw("Failure allocating memory for cLNUMBER");
		delete[] pData;
		pData = p2;
		uNoOfAtoms = Num.uNoOfAtoms;
	}
	memset((void*)pData, 0, uNoOfAtoms*sizeof(unsigned long));
	memcpy((void*)pData, (void*)Num.pData, uNoOfAtoms*sizeof(unsigned long));
	if (Num.bGarbage) {
#ifdef DEBUG
		printf("destroying the other after assignment to it...\n");
#endif
		Num.ReleaseMemory();
	}
	return *this;
}

cLNUMBER& cLNUMBER::operator=(double Num)  throw(char*){
	int sign = Num > 0 ? 1 : -1;
	if (sign == -1) Num = -Num;
	for (unsigned i=0; i<uNoOfAtoms-1; i++) {
		pData[i]=(unsigned long)Num;
		Num = (Num-pData[i])*BASE; // shift Num NOOFDIGS decimal digits left - MM
	}
	if (sign == -1) Negate();
	return *this;
}


void cLNUMBER::Negate() {
	if (!uNoOfAtoms || !pData) return;
	unsigned int i=uNoOfAtoms-1, j=0;

	while (i>0 && pData[i]==0) i--; // skip zeroes at the end - MM
	if (i) {
		pData[i]=BASE-pData[i];
		j=i-1;
		while (j) {pData[j]=BASE-1-pData[j];j--;}
		pData[0] = 2*BASE-1-pData[0];
	}
	else if (pData[0]) pData[0] = 2*BASE-pData[0]; // only the integer part is non-zero - MM
}

void cLNUMBER::print() const {
	unsigned int i = 0;
	if (uNoOfAtoms && pData) 
		if (iSign() == -1) {
			cLNUMBER ctemp(*this);
			ctemp.Negate();
			printf("-");ctemp.print();
			ctemp.~cLNUMBER();ctemp.pData = NULL;
		}
		else {
#ifdef LONGBASE
			printf("%06lu.",pData[0]);
			for (i=1;i<uNoOfAtoms;i++) printf("%06lu",pData[i]);
#else
			printf("%04lu.",pData[0]);
			for (i=1;i<uNoOfAtoms;i++) printf("%04lu",pData[i]);
#endif
		}
}

// for manual garbage collection of temp objects not on stack - MM 
void cLNUMBER::ReleaseMemory() {
	if (pData) {
		free(pData);
		pData = NULL;
	}
}

bool  cLNUMBER::bFromString(char * pStr) {
	unsigned int i = 1, j = 0;
	int iIntPart = 0;
	char * pDecimalPnt  = strchr(pStr, '.');
		
	// read the integral part if a decimal point is present and is not the first char - MM
	if (pDecimalPnt && pDecimalPnt != pStr) {
		// non-empty integral part - MM
		sscanf(pStr, "%d.", &iIntPart);
	}
	char * pFracPart = NULL;
	
	// if fractional part is present - MM
	if (pDecimalPnt && pDecimalPnt-pStr+1<(int)strlen(pStr)) {
		pFracPart =  pDecimalPnt; 
		pFracPart++;
	}

	if (!uNoOfAtoms) return true;

	unsigned int len = strlen(pFracPart);
	if (pFracPart)
		while (i<uNoOfAtoms) {
			for (int k=0; k< NOOFDIGS; k++) 
				if (j<len && pFracPart[j]>='0' && pFracPart[j]<='9')  {
					pData[i] = pData[i]*10+int(pFracPart[j++]-'0');
				}
			i++;
		}
	
	// integral part - MM
	pData[0]=iIntPart;
	if (iIntPart < 0) { // negative - MM
		pData[0]=-pData[0];
		Negate();
	}

	return true;
}

cLNUMBER::~cLNUMBER() {
	ReleaseMemory();
}
