/***************************************************************************
 *   Calculation of Pi to arbitrary decimal places			   *
 *   pi.cpp - master file                                                   *
 *                                                                         *
 *   Copyright (C) 2005 by Musa A. Maharramov   			   *
 *                                                                         *
 *   Department of Applied Mathematics and Cybernetics			   * 
 *   Baku State University						   *
 *   musa@maharramov.com						   *
 *   www.maharramov.com							   *
 *						  			   *
 *   Permission is hereby granted, free of charge, to any person obtaining *
 *   a copy of this software and associated documentation files (the       *
 *   "Software"), to deal in the Software without restriction, including   *
 *   without limitation the rights to use, copy, modify, merge, publish,   *
 *   distribute, sublicense, and/or sell copies of the Software, and to    *
 *   permit persons to whom the Software is furnished to do so, subject to *
 *   the following conditions:                                             *
 *                                                                         *
 *   The above copyright notice and this permission notice shall be        *
 *   included in all copies or substantial portions of the Software.       *
 *                                                                         *
 *   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,       *
 *   EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF    *
 *   MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.*
 *   IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR     *
 *   OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, *
 *   ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR *
 *   OTHER DEALINGS IN THE SOFTWARE.                                       *
 ***************************************************************************/

/*
	pi.cpp - the master file of the program calculating Pi
		or a specified part of Machin's series
	
	Other project files include:

	pi.h - master header file
	strings.cpp - separately compiled string message file
		messagestrings.h - corresponding declaration file 
	longmath.h - high precision arithmetic header file
	longmath.cpp - separately compiled high precision arithmetic library
	longadd.cpp - a standalone program for adding two high-precision numbers
	calcpi.pl - a Perl script for automatically distributing the task across
		multiple systems
	ehosts.txt - a text file with a list of hosts to be used for performing 
		 the calculation (similar to e.g. LAM schema file)
	Makefile - make file for GNU make
 
*/

// the following include file is ignored for UNIX-like systems
// and is required for the generation of Microsoft pre-compiled headers - MM

#ifdef _WIN32			
	#include "stdafx.h"
#endif

// include master header file - MM
#include "pi.h"

// length of string variables/buffers - MM
#define MAXCHARS 512

unsigned int 
	// default no of decimal places - MM
	uNoOfDigitsInLongNumber = 100, 
	// default no of terms of the series to sum - MM	
	uNoOfTermsInSeries = 0,
	// index of the initial term of the series - MM	   
	uInitialIndex = 0;			   

bool
	// boolean flag indicating that the program should calculate the number of terms of the series 
	// that is required to sum up to ensure the requested accuracy; output the result and exit  - MM
	bCalcRequiredTerms = false;	   
	
// estimate the number of terms of Machin's Series required for 
// ensuring the required accuracy of 'digs' decimal places - MM
unsigned int uNoOfTemsFromAccuracy(unsigned int digs) {
	// meaningful accuracy is from 4 to 100 million decimal places - MM 
	if (digs < 4 || digs > 100000000) return 0;
	unsigned int k=1;
	k = (unsigned int)((digs*log(10)+log(32)-log(5))/(2*log(5)+1)+1);
	//while ((2*k+1)*log(5)-log(32)+log(2*k+1)-digs*log(10)<=0) k++;
	return k;
}


// command-line parameters - MM 
const char 
	// prefix for the required decimal places - MM 
	szcRequiredDecimaDigits[] = "-d\0",
	// prefix for the required start index - MM
	szcStartIndex[]="-s\0",
	// prefix for the number of terms of Machin's Series to be summed - MM
	szcNoOfTermsToSum[]="-n\0",
	// command-line parameter indicating that the program should calculate the number of 
	// terms of the series that is required to be summed up to ensure the requested accuracy
	// using the function 'uNoOfTemsFromAccuracy', then output the result and exit  - MM
	szcNoOfTermsRequired[]="-r\0";

// parse command-line parameters - MM
bool bReadParams(int argc, char* argv[]) {
	int i = 1;
	while (i<argc) {
		if (strcmp(argv[i], szcRequiredDecimaDigits) == 0)  {
			i++; if (i==argc) return false;
			int digs = 0;
			sscanf(argv[i],"%d", &digs);
			// meaningful accuracy is from 4 to 100 million decimal places - MM 
			if  (digs < 4 || digs > 100000000) {
				printf(szcMessageDigitsRange);
				return false;
			}
			uNoOfDigitsInLongNumber = (unsigned int)digs;
			i++;
		}
		else if (strcmp(argv[i], szcStartIndex) == 0)  {
			i++; if (i==argc) return false;
			int iIndex = 0;
			sscanf(argv[i],"%d", &iIndex);
			if  (iIndex < 0) {
				printf(szcMessageInitialTermIndex);
				return false;
			}
			uInitialIndex = (unsigned int)iIndex;
			i++;
		}
		else if (strcmp(argv[i], szcNoOfTermsToSum) == 0)  {
			i++; if (i==argc) return false;
			int iNo = 0;
			sscanf(argv[i],"%d", &iNo);
			if  (iNo <= 0) {
				printf(szcMessageNoOfTerms);
				return false;
			}
			uNoOfTermsInSeries = (unsigned int)iNo;
			i++;
		}
		else if (strcmp(argv[i], szcNoOfTermsRequired) == 0)  {
			bCalcRequiredTerms = true;
			i++;
		}
		else return false;
	}
	return true;
}


void printusage() {
	printf(szcMessageHelp);
}

int main(int argc, char* argv[])
{

	if (!bReadParams(argc, argv)) {
		printusage();
		return 1;
	}

	if (bCalcRequiredTerms) {
		printf("%u\n", uNoOfTemsFromAccuracy(uNoOfDigitsInLongNumber));
		return 0;
	}

	if (!uNoOfTermsInSeries) uNoOfTermsInSeries =uNoOfTemsFromAccuracy(uNoOfDigitsInLongNumber);

	cLNUMBER* pA, *pB, *pC;

	pA = new cLNUMBER(uNoOfDigitsInLongNumber);
	pB = new cLNUMBER(uNoOfDigitsInLongNumber);
	pC = new cLNUMBER(uNoOfDigitsInLongNumber);

	*pA = 16;
	*pB = 4;
	*pA = *pA / 5;
	*pB = *pB / 239;

	for (unsigned int i = 0; i <uInitialIndex; i++) {
		*pA = *pA/25;
		*pB = *pB/239;*pB = *pB/239;
	}

	for (unsigned int i = uInitialIndex; i <uInitialIndex+uNoOfTermsInSeries; i++) {
		if ((i % 2) == 0) *pC = *pC + (*pA-*pB)/(2*i+1);
		else *pC = *pC - (*pA-*pB)/(2*i+1);
		*pA = *pA/25;
		*pB = *pB/239;*pB = *pB/239;
	}

	pC->print();
	printf("\n");

	delete pA; delete pB; delete pC;

	return 0;
}
