C/***************************************************************************
C *   Parallel Simulation of an Endocrine Organ
C *
C *   (see http://www.mamedova.com/072006.pdf)
C *
C *   Copyright (C) 2006 Mehriban V. Mamedova
C *
C *   Institute of Cybernetics
C *   Azerbaijan National Academy
C *   mehriban@mamedova.com
C *   www.mamedova.com
C *
C *   Permission is hereby granted, free of charge, to any person obtaining *
C *   a copy of this software and associated documentation files (the       *
C *   "Software"), to deal in the Software without restriction, including   *
C *   without limitation the rights to use, copy, modify, merge, publish,   *
C *   distribute, sublicense, and/or sell copies of the Software, and to    *
C *   permit persons to whom the Software is furnished to do so, subject to *
C *   the following conditions:                                             *
C *                                                                         *
C *   The above copyright notice and this permission notice shall be        *
C *   included in all copies or substantial portions of the Software.       *
C *                                                                         *
C *   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,       *
C *   EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF    *
C *   MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.*
C *   IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR     *
C *   OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, *
C *   ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR *
C *   OTHER DEALINGS IN THE SOFTWARE.                                       *
C***************************************************************************/

C
C     Моделирование функционирования желез внутренней секреции 
C     гранулярного типа
C
C     (С) 2006 Мехрибан Мамедова
C
C     ЗАПУСК В ОС UNIX: gland < "файл с входными данными"
C
C     Результаты моделирования записываются в стандартный выходной поток
C
      PROGRAM GLAND
C
C
	IMPLICIT NONE
C
C
	CHARACTER*80 TITLE
C
C     Количество моделируемых клеток
C
      INTEGER N
C
C     Максимальная скорость синтеза гормона
C
      REAL VX_MAX
C
C     Максимальная концентрация гормона в клетке
C
      REAL X_MAX
C
C     Экспоненциальный параметр скорости синтеза гормона
C
      REAL VX_ALPHA
C
C     Увеличение концентрации побочных продуктов при секреции
C
      REAL YC
C     
C     Коэффициент пропорциониальности 
C     скорости синтеза побочных веществ
C     скорости синтеза гормона в фазе синтеза 
C
      REAL VY_TO_VX_SYNTH
C     
C     Постоянная скорость синтеза побочных веществ
C     в фазе готовности к секреции 
C
      REAL VY_READY_SECR
C     
C     Постоянная скорость выведения побочных веществ
C     в фазе восстановления 
C
      REAL VY_RECOVERY
C     
C     Коэффициент выведения (побочных веществ)
C
      REAL MU
C
C     Усреднённые периоды генерации сигналов секреции и синтеза
C
      REAL MEAN_SECR_SIGNAL_DELTA, MEAN_SYNTH_SIGNAL_DELTA
C
C     Длительность сигналов секреции и синтеза
C
      REAL SECR_SIGNAL_LEN, SYNTH_SIGNAL_LEN
C
C     Максимальный период моделирования
C
      REAL MAX_T
C
C     Шаг времени моделирования
C
      REAL DT
C
C     Максимальное время пребывания в фазе подготовки к секреции,
C     по истечении которого гормон в клетке разрушается
C
      REAL MAX_READY_PHASE
C
C     Случайное распределение точки отсчёта для моделирования
C     индивидуальной клетки
C
      REAL T_INITIAL
C
C     Максимальные уклонения параметров от нормы (>0,<1)
C
      REAL MAXDEV_VX_MAX, MAXDEV_X_MAX, MAXDEV_VX_ALPHA 
      REAL MAXDEV_YC, MAXDEV_VY_READY_SECR
      REAL MAXDEV_VY_RECOVERY, MAXDEV_MU,MAXDEV_VY_TO_VX_SYNTH
      REAL MAXDEV_SECR_SIGNAL_DELTA, MAXDEV_SYNTH_SIGNAL_DELTA
      REAL MAXDEV_MAX_READY_PHASE
      REAL MAXDEV_SECR_SIGNAL_LEN,MAXDEV_SYNTH_SIGNAL_LEN
C
	TITLE='MODELLING OF GRANULATED SECRETION GLANDS'
	WRITE (*,*) TITLE

      READ (*,*) N
C
C     Прочитать все параметры из стандартного входного потока
C      
      READ (*,*) VX_MAX,X_MAX,VX_ALPHA,YC,
     1 VY_TO_VX_SYNTH,VY_READY_SECR,VY_RECOVERY,MU,
     2 MEAN_SECR_SIGNAL_DELTA,MEAN_SYNTH_SIGNAL_DELTA,
     3 SECR_SIGNAL_LEN, SYNTH_SIGNAL_LEN,MAX_T,DT,
     4 MAX_READY_PHASE,T_INITIAL,MAXDEV_VX_MAX,MAXDEV_X_MAX,
     5 MAXDEV_VX_ALPHA,MAXDEV_YC,MAXDEV_VY_READY_SECR,
     6 MAXDEV_VY_RECOVERY, MAXDEV_MU,MAXDEV_VY_TO_VX_SYNTH,
     7 MAXDEV_SECR_SIGNAL_DELTA,MAXDEV_SYNTH_SIGNAL_DELTA,
     8 MAXDEV_MAX_READY_PHASE,MAXDEV_SECR_SIGNAL_LEN,
     9 MAXDEV_SYNTH_SIGNAL_LEN
 
C
C     Вызов подпрограммы моделирования
C
	CALL MODEL(N,VX_MAX,X_MAX,VX_ALPHA,YC,
     1 VY_TO_VX_SYNTH,VY_READY_SECR,VY_RECOVERY,MU,
     2 MEAN_SECR_SIGNAL_DELTA,MEAN_SYNTH_SIGNAL_DELTA,
     3 SECR_SIGNAL_LEN, SYNTH_SIGNAL_LEN,MAX_T,DT,
     4 MAX_READY_PHASE,T_INITIAL,MAXDEV_VX_MAX,MAXDEV_X_MAX,
     5 MAXDEV_VX_ALPHA,MAXDEV_YC,MAXDEV_VY_READY_SECR,
     6 MAXDEV_VY_RECOVERY, MAXDEV_MU,MAXDEV_VY_TO_VX_SYNTH,
     7 MAXDEV_SECR_SIGNAL_DELTA,MAXDEV_SYNTH_SIGNAL_DELTA,
     8 MAXDEV_MAX_READY_PHASE,MAXDEV_SECR_SIGNAL_LEN,
     9 MAXDEV_SYNTH_SIGNAL_LEN)
C
	STOP
	END
C
C
C
	SUBROUTINE MODEL(N,VX_MAX,X_MAX,VX_ALPHA,YC,
     1 VY_TO_VX_SYNTH,VY_READY_SECR,VY_RECOVERY,MU,
     2 MEAN_SECR_SIGNAL_DELTA,MEAN_SYNTH_SIGNAL_DELTA,
     3 SECR_SIGNAL_LEN, SYNTH_SIGNAL_LEN,MAX_T,DT,
     4 MAX_READY_PHASE,T_INITIAL,MAXDEV_VX_MAX,MAXDEV_X_MAX,
     5 MAXDEV_VX_ALPHA,MAXDEV_YC,MAXDEV_VY_READY_SECR,
     6 MAXDEV_VY_RECOVERY, MAXDEV_MU,MAXDEV_VY_TO_VX_SYNTH,
     7 MAXDEV_SECR_SIGNAL_DELTA,MAXDEV_SYNTH_SIGNAL_DELTA,
     8 MAXDEV_MAX_READY_PHASE,MAXDEV_SECR_SIGNAL_LEN,
     9 MAXDEV_SYNTH_SIGNAL_LEN)
C
	IMPLICIT NONE
C
C     ***** ПАРАМЕТРЫ ПОДПРОГРАММЫ *****
C
C     Количество моделируемых клеток
C
      INTEGER N
C
C     Максимальная скорость синтеза гормона
C
      REAL VX_MAX
C
C     Максимальная концентрация гормона в клетке
C
      REAL X_MAX
C
C     Экспоненциальный параметр скорости синтеза гормона
C
      REAL VX_ALPHA
C
C     Увеличение концентрации побочных продуктов при секреции
C
      REAL YC
C     
C     Коэффициент пропорциониальности 
C     скорости синтеза побочных веществ
C     скорости синтеза гормона в фазе синтеза 
C
      REAL VY_TO_VX_SYNTH
C     
C     Постоянная скорость синтеза побочных веществ
C     в фазе готовности к секреции 
C
      REAL VY_READY_SECR
C     
C     Постоянная скорость выведения побочных веществ
C     в фазе восстановления 
C
      REAL VY_RECOVERY
C     
C     Коэффициент выведения (побочных веществ)
C
      REAL MU
C
C     Усреднённые периоды генерации сигналов секреции и синтеза
C
      REAL MEAN_SECR_SIGNAL_DELTA, MEAN_SYNTH_SIGNAL_DELTA
C
C     Длительность сигналов секреции и синтеза
C
      REAL SECR_SIGNAL_LEN, SYNTH_SIGNAL_LEN
C
C     Максимальный период моделирования
C
      REAL MAX_T
C
C     Шаг времени моделирования
C
      REAL DT
C
C     Максимальное время пребывания в фазе подготовки к секреции,
C     по истечении которого гормон в клетке разрушается
C
      REAL MAX_READY_PHASE
C
C     Случайное распределение точки отсчёта для моделирования
C     индивидуальной клетки
C
      REAL T_INITIAL
C
C     Максимальные уклонения параметров от нормы (>0,<1)
C
      REAL MAXDEV_VX_MAX, MAXDEV_X_MAX, MAXDEV_VX_ALPHA 
      REAL MAXDEV_YC, MAXDEV_VY_READY_SECR
      REAL MAXDEV_VY_RECOVERY, MAXDEV_MU,MAXDEV_VY_TO_VX_SYNTH
      REAL MAXDEV_SECR_SIGNAL_DELTA, MAXDEV_SYNTH_SIGNAL_DELTA
      REAL MAXDEV_MAX_READY_PHASE
      REAL MAXDEV_SECR_SIGNAL_LEN,MAXDEV_SYNTH_SIGNAL_LEN
C     
C     ***** ПЕРЕМЕННЫЕ ПОДПРОГРАММЫ *****
C
C     Время     
C     
      REAL T
C
C     Концентрации гормона и побочных продуктов в каждой клетке
C
      REAL X(N),Y(N)

C
C     Фаза каждой клетки
C
C     0 - ожидание между восстановлением и синтезом
C     1 - синтез
C     2 - подготовка к секреции 
C     3 - восстановление после секреции
C
      INTEGER PHASE(N)
C
C     Параметры A и B экспоненциальной зависимости 
C     скорости синтеза гормона от концентрации в отдельной клетке
C     VX = A*exp(-alpha*concentration)-B
C
      REAL A(N), B(N)
C
C     Максимальная концентрация гормона в клетке
C
      REAL CELL_X_MAX(N)
C
C     Экспоненциальный параметр скорости синтеза гормона
C
      REAL CELL_VX_ALPHA(N)
C
C     Увеличение концентрации побочных продуктов при секреции
C
      REAL CELL_YC(N)
C     
C     Коэффициент пропорциониальности 
C     скорости синтеза побочных веществ
C     скорости синтеза гормона в фазе синтеза 
C
      REAL CELL_VY_TO_VX_SYNTH(N)
C     
C     Постоянная скорость синтеза побочных веществ
C     в фазе готовности к секреции 
C
      REAL CELL_VY_READY_SECR(N)
C     
C     Постоянная скорость выведения побочных веществ
C     в фазе восстановления 
C
      REAL CELL_VY_RECOVERY(N)
C     
C     Коэффициент выведения (побочных веществ)
C
      REAL CELL_MU(N)
C
C     Максимальное время пребывания в фазе подготовки к секреции,
C     по истечении которого гормон в клетке разрушается
C
      REAL CELL_MAX_READY_PHASE(N)
C
C     Время пребывания каждой клетки в фазе подготовки к секреции
C
      REAL CELL_READY_PHASE(N)

C
C     Переменные, используеиые в моделировании посылки и 
C     продолжительности сигналов секреции и синтеза
C
      REAL TIME_TO_SECR_SIGNAL, TIME_TO_SYNTH_SIGNAL
      REAL DURATION_SECR_SIGNAL, DURATION_SYNTH_SIGNAL
C
C     Количество гормона, выбрасываемое в данный 
C     промежуток времени DT
C
      REAL HORMONE
C
C     Общее количество гормона и побочных веществ во всех
C     клетках в данный момент времени
C
      REAL TOTAL_X, TOTAL_Y
C
C
C     Вспомогательные переменные
C      
      INTEGER I
      REAL TEMP
C
C     FUNCTION
C
      REAL RAND

C
C     Инициализация начального состояния органа,
C     синтезирующего гормоны
C
!$OMP PARALLEL DO PRIVATE(I,TEMP)
      DO I=1,N
        CELL_X_MAX(I) = X_MAX*(1. +  MAXDEV_X_MAX*RAND(0.))
C
C       Рассчитать параметры A и B экспоненциальной зависимости 
C       скорости синтеза гормона от концентрации
C
        TEMP = VX_MAX*(1. +  MAXDEV_VX_MAX*RAND(0.))
        CELL_VX_ALPHA(I)= VX_ALPHA*(1. +  MAXDEV_VX_ALPHA*RAND(0.))
        A(I)=TEMP/(1.-EXP(-CELL_VX_ALPHA(I)*CELL_X_MAX(I)))
        B(I)=A(I) - TEMP
C
C       и прочие индивидуальные переметры функционирования
C       каждой клетки
C
C       Сущность применения метода Монте-Карло в данной задаче
C       сводится к случайному выбору параметров начального состояния
C       и фазы каждой клетки
C
        CELL_YC(I) = YC*(1. +  MAXDEV_YC*RAND(0.))
        CELL_VY_TO_VX_SYNTH(I) = VY_TO_VX_SYNTH*(1. +  
     1      MAXDEV_VY_TO_VX_SYNTH*RAND(0.))
        CELL_VY_READY_SECR(I) = VY_READY_SECR*(1. +  
     1      MAXDEV_VY_READY_SECR*RAND(0.))
        CELL_VY_RECOVERY(I) = VY_RECOVERY*(1. +  
     1      MAXDEV_VY_RECOVERY*RAND(0.))
        CELL_MU(I) = MU*(1. +  MAXDEV_MU*RAND(0.))
        CELL_MAX_READY_PHASE(I)= MAX_READY_PHASE*(1. +  
     1      MAXDEV_MAX_READY_PHASE*RAND(0.))
C
C       Начальное случайное распределение состояний клеток
C
        PHASE(I)=RAND(0.)*4.
        IF (PHASE(I). GT. 3) THEN
          PHASE(I)=0.
        END IF
        IF (PHASE (I) .EQ. 0) THEN
C         Состояние ожидания после восстановления
          X(I)=0.
          Y(I)=0.
          CELL_READY_PHASE(I)=0.
        ELSEIF (PHASE (I) .EQ. 1) THEN
C         Синтез
          X(I)=CELL_X_MAX(I)*RAND(0.)
          Y(I)=CELL_VY_TO_VX_SYNTH(I)*X(I)
          CELL_READY_PHASE(I)=0.
        ELSEIF (PHASE (I) .EQ. 2) THEN
C         Подготовка к секреции
          X(I)=CELL_X_MAX(I)
          CELL_READY_PHASE(I) = RAND(0.)*CELL_MAX_READY_PHASE(I)
          Y(I)=X(I)*CELL_VY_TO_VX_SYNTH(I)+
     1      CELL_READY_PHASE(I)*CELL_VY_READY_SECR(I)
        ELSE
C         Восстановление
          X(I)=0.
          Y(I)=CELL_X_MAX(I)*CELL_VY_TO_VX_SYNTH(I)+
     1      RAND(0.)*CELL_MAX_READY_PHASE(I)*CELL_VY_READY_SECR(I)-
     2      RAND(0.)*CELL_MAX_READY_PHASE(I)*CELL_VY_RECOVERY(I)
          CELL_READY_PHASE(I)=0.     
        ENDIF
      END DO      
!$OMP END PARALLEL DO
C
C     Время
C
      T=0.
C
C     Проинициализировать (псевдослучайные) периоды посылки сигналов
C     синтеза и секреции, а также их продолжительность
C
      TIME_TO_SECR_SIGNAL = MEAN_SECR_SIGNAL_DELTA*
     1      (1.+MAXDEV_SECR_SIGNAL_DELTA*RAND(0.))
      TIME_TO_SYNTH_SIGNAL = MEAN_SYNTH_SIGNAL_DELTA*
     1      (1.+MAXDEV_SYNTH_SIGNAL_DELTA*RAND(0.))
     
      DURATION_SECR_SIGNAL = 0.
      DURATION_SYNTH_SIGNAL = 0.    
C
C     Основной цикл моделирования
C
      DO WHILE (T .LT. MAX_T)
        T = T + DT
C       Количество гормона, выбрасываемое в данный 
C       промежуток времени DT
        HORMONE = 0.
C
C       Проверить необходимость выброса гормона
        IF (TIME_TO_SECR_SIGNAL .GT. 0.) THEN
C         Сигнал секреции ещё не поступил, продолжаем ожидание...
          TIME_TO_SECR_SIGNAL = TIME_TO_SECR_SIGNAL - DT
        ELSE
          IF (DURATION_SECR_SIGNAL .GT. 0) THEN
C           Клетки продолжают получать сигнал о секреции
            DURATION_SECR_SIGNAL = DURATION_SECR_SIGNAL - DT
C           Выбросить гормон...
            DO I=1,N
              IF ((PHASE (I) .EQ. 2) .OR. (PHASE (I) .EQ. 1)) THEN
C               Выпустить гормон в кровь
C               и перейти в состояние восстановления,
C               если клетка находилась в состоянии синтеза или подготовки
C               к секреции
                HORMONE = HORMONE + X(I)
                X(I)=0.
                Y(I)=Y(I)+CELL_YC(I)
                PHASE (I) = 3
                CELL_READY_PHASE(I) = 0.
              END IF
            END DO
          ELSE
C           Сигнал о секреции прекратился - возобновить отсчёт
C           времени до следующего сигнала секреции
            TIME_TO_SECR_SIGNAL = MEAN_SECR_SIGNAL_DELTA*
     1        (1.+MAXDEV_SECR_SIGNAL_DELTA*RAND(0.))
            DURATION_SECR_SIGNAL = SECR_SIGNAL_LEN*
     1        (1.+MAXDEV_SECR_SIGNAL_LEN*RAND(0.))
          END IF
        END IF
C
C       Проверить необходимость синтеза гормона
        IF (TIME_TO_SYNTH_SIGNAL .GT. 0.) THEN
C       Сигнал синтеза ещё не поступил, продолжаем ожидание...
          TIME_TO_SYNTH_SIGNAL = TIME_TO_SYNTH_SIGNAL - DT
        ELSE
          IF (DURATION_SYNTH_SIGNAL .GT. 0) THEN
C           Клетки продолжают получать сигнал о синтезе
            DURATION_SYNTH_SIGNAL = DURATION_SYNTH_SIGNAL - DT
            DO I=1,N
              IF (PHASE (I) .EQ. 0) THEN
C               Перейти в фазу синтеза, если кетка находится 
C               в ожидании...
                PHASE(I)=1
              END IF
            END DO
          ELSE
C         Сигнал о синтезе прекратился - возобновить отсчёт
C         времени до следующего сигнала синтеза

            TIME_TO_SYNTH_SIGNAL = MEAN_SYNTH_SIGNAL_DELTA*
     1        (1.+MAXDEV_SYNTH_SIGNAL_DELTA*RAND(0.))
            DURATION_SYNTH_SIGNAL = SYNTH_SIGNAL_LEN*
     1        (1.+MAXDEV_SYNTH_SIGNAL_LEN*RAND(0.))
          END IF
        END IF

!$OMP PARALLEL DO PRIVATE(I,TEMP)
        DO I=1,N
        
          IF (PHASE (I) .EQ. 1) THEN
C           Синтез
            TEMP = A(I)*EXP(-CELL_VX_ALPHA(I)*X(I))-B(I)
            X(I)=X(I)+DT*TEMP
            Y(I)=Y(I)+DT*CELL_VY_TO_VX_SYNTH(I)*TEMP
            IF (X(I) .GE. CELL_X_MAX(I)) THEN
C             Перейти в состояние готовности к секреции
              PHASE(I) = 2
              CELL_READY_PHASE(I) = 0
            END IF

          ELSEIF (PHASE (I) .EQ. 2) THEN
C           Подготовка к секреции
            X(I)=CELL_X_MAX(I)
            Y(I)=Y(I)+DT*CELL_VY_READY_SECR(I)
            CELL_READY_PHASE(I)=CELL_READY_PHASE(I)+DT
            IF (CELL_READY_PHASE(I) .GE. CELL_MAX_READY_PHASE(I)) THEN
C             Уничтожить гормон в отсутствии сигнала секреции
C             и перейти в состояние восстановления
              X(I)=0.
              Y(I)=Y(I)+CELL_YC(I)
              PHASE (I) = 3
              CELL_READY_PHASE(I) = 0.
            END IF
          ELSEIF (PHASE (I) .EQ. 3) THEN
C           Восстановление
            X(I)=0.
            IF (Y(I) .GE. CELL_VY_RECOVERY(I)/CELL_MU(I)) THEN
              Y(I)=Y(I) - DT*CELL_VY_RECOVERY(I)
            ELSE
              Y(I)=Y(I) - DT*CELL_VY_RECOVERY(I)
            END IF

            IF (Y(I) .LE. 0.) THEN
C             Восстановление завершено - перейти в состояние
C             ожидания сигнала синтеза
              Y(I) = 0.
              PHASE(I) = 0
            END IF
          ENDIF
        END DO
!$OMP END PARALLEL DO

        TOTAL_X = 0.
        TOTAL_Y = 0.
        DO I=1,N
          TOTAL_X = TOTAL_X + X(I)
          TOTAL_Y = TOTAL_Y + Y(I)
        END DO
C
C       Вывод в стандартный выходной поток
C
        WRITE (*,*) T,' ',TOTAL_X,' ',TOTAL_Y,' ',HORMONE

      END DO        
C
	RETURN
	END
